﻿using ProbitConstants.DALI.Constants2;
using ProbitConstants.MacroControl;
using ProbitConstants.MacroInterface;
using System.Collections.Generic;
using System.Drawing;
using System.Globalization;
using System.Threading;
using System.Windows.Forms;

namespace BasicMacro
{
    public partial class cBasicMacroControl : aMacroControl
    {
        #region basic methods inherited by aMacroControl

        public cBasicMacroControl()
        {
            InitializeComponent();
            
            if (!this.IsHandleCreated)
            {
                try
                {
                    this.CreateHandle();
                }
                catch
                {
                }
            }

            //DO CONSTRUCTOR STUFF HERE
        }

        public override void Init()
        {
            //DO INIT STUFF HERE
        }
        public override void Uninit()
        {
            //DO UNINIT STUFF HERE
        }

        #endregion

        #region example: control specific functions for macro1, macro2 and macro3
        
        private void Macro1Button_Click(object sender, System.EventArgs e)
        {
            if (MacroInterface != null)
            {
                //Example: Create a list/container for the commands which should be transmitted via macro interface
                List<cCommandElementBase> macro = new List<cCommandElementBase>();

                //Example: Using additional data
                macro.Add(new cCommandElement2(eDALICommand2.OFF, null)); //OFF command has no additional data, therefore it will be ignored anyway (but we set it here to null explicitly)
                macro.Add(new cCommandElement2(eDALICommand2.DAPC, 254)); //DAPC command uses additional data (here for arc power level)


                //Example: Using address information
                macro.Add(new cCommandElement2(eDALICommand2.DAPC, 254));                                                                          //uses the global selected address given by the ProbitBench
                macro.Add(new cCommandElement2(eDALICommand2.GO_TO_SCENE_0, null, new cAddressInformation2(eDeviceAddressType.GROUP_ADDRESS, 2))); //uses the explicit address given the cAddressInformation object, here group 2
                macro.Add(new cCommandElement2(eDALICommand2.DTR0, 128));                                                                          //uses the global selected address given by the ProbitBench, but in this case DTR0 is a broadcast command anyway
                macro.Add(new cCommandElement2(eDALICommand2.SET_POWER_ON_LEVEL, null, (cAddressInformation2)null));                               //uses the global selected address given by the ProbitBench, by setting cAddressInformation object to null explicitly


                //Example: Using transmit information
                macro.Add(new cCommandElement2(eDALICommand2.DTR0, 250));                                                                    //DTR0 is a non-configuration command by default and will be transmitted once only
                macro.Add(new cCommandElement2(eDALICommand2.SET_MIN_LEVEL, null));                                                          //SET MIN LEVEL command is a send twice command by default and will be transmitted twice with default timings in between
                macro.Add(new cCommandElement2(eDALICommand2.SET_POWER_ON_LEVEL, null, null, new cTransmitInformation(null, 0, null, 150000, 2))); //SET_POWER_ON_LEVEL command is transmitted explicitly twice with 150000µs=150ms settling time in between


                //Example: Send created macro list/container via macro interface
                MacroInterface.SendMacro(macro); //non-blocking call
            }
        }
        private void Macro2Button_Click(object sender, System.EventArgs e)
        {
            if (MacroInterface != null)
            {
                List<cCommandElementBase> macro = new List<cCommandElementBase>();

                //Example: Mixing different command types (102 and 103)
                macro.Add(new cCommandElement2(eDALICommand2.DAPC, 254));                       //using 102 commands with global selected address given by the ProbitBench
                macro.Add(new cCommandElement3(eDALICommand3.ADD_TO_DEVICE_GROUPS_0_15, 0xFF)); //using 103 device commands with global selected address given by the ProbitBench
                macro.Add(new cCommandElement4(eDALICommand4.ENABLE_INSTANCE, null, new cAddressInformation4(eDeviceAddressType.BROADCAST_UNADDRESSED, null, eInstanceAddressType.INSTANCE_BROADCAST, null))); //using 103 instance commands with explicit instance command addressing
                macro.Add(new cCommandElement5(eDALICommand5.OPCODE, 0, new cAddressInformation5(eDeviceAddressType.GROUP_ADDRESS, 0, eFeatureAddressType.FEATURE_DEVICE_LEVEL, null)));                       //using 103 feature commands with explicit feature command addressing
                macro.Add(new cCommandElement6(eDALICommand6.POWER_NOTIFICATION, 255));   //using 103 events with implicit (by command) event addressing
                
                MacroInterface.SendMacro(macro); //non-blocking call
            }
        }
        private void Macro3Button_Click(object sender, System.EventArgs e)
        {
            AnswerTextBox.Text = "";

            if (MacroInterface != null)
            {
                List<cCommandElementBase> macro = new List<cCommandElementBase>();

                macro.Add(new cCommandElement2(eDALICommand2.QUERY_POWER_ON_LEVEL, null)); //uses the global selected address given by the ProbitBench

                //Example: Send created macro list/container via macro interface but wait for answer
                cMacroResult result = MacroInterface.SendMacroAndWaitForAnswer(macro); //blocking call

                //Example: Evaluate last answer of macro transmission operation and set AnswerTextBox to the answer content 
                if (result != null)
                {
                    switch (result.m_result)
                    {
                        case eMacroResult.ANSWER_VALUE:
                            AnswerTextBox.Text = result.m_answer_value.ToString();
                            break;

                        case eMacroResult.ANSWER_TIMEOUT:
                            AnswerTextBox.Text = "timeout";
                            break;

                        case eMacroResult.ANSWER_FRAMING_ERROR:
                            AnswerTextBox.Text = "framing error";
                            break;

                        case eMacroResult.EXECUTION_FAILED:
                        case eMacroResult.UNKNOWN:
                            AnswerTextBox.Text = "interface error";
                            break;

                        default:
                            break;
                    }
                }
            }

        }

        #endregion

        #region example: control specific functions for macro4 (by using a thread)

        private void Macro4Button_Click(object sender, System.EventArgs e)
        {
            //To realise a more complex process flow (especially when using Thread.Sleep(...) to wait a specific time)
            //it is recommended to decouple transceiving part (e.g. sending macros or switching) from GUI part in a separate thread otherwise GUI could be blocked.
            //Therefore this function just starts the thread (which executes Macro4Thread()) but does not wait for its end.

            //create macro thread
            Thread macro_thread = new Thread(Macro4Thread);
            macro_thread.IsBackground = true;
            macro_thread.Priority = ThreadPriority.Normal;
            macro_thread.CurrentCulture = CultureInfo.InvariantCulture;
            macro_thread.CurrentUICulture = CultureInfo.InvariantCulture;
            
            //start macro thread
            macro_thread.Start();
        }

        private void Macro4Thread()
        {
            if (MacroInterface != null)
            {
                List<cCommandElementBase> macro = new List<cCommandElementBase>();
                cMacroResult result = null;

                //Set all switches to default operation
                MacroInterface.SetOutput(eOutputTarget.DALI_BUS_POWER, true); //enable DALI BPS
                MacroInterface.SetOutput(eOutputTarget.DALI_TERMINAL_DUT_CONNECTED, true); //connect DALI bus to DUT
                MacroInterface.SetOutput(eOutputTarget.DALI_TERMINAL_POLARITY_INVERTED, false); //disable DALI bus terminal invertion
                MacroInterface.SetOutput(eOutputTarget.DIGITAL_SWITCH_0, true); //connect lamp to DUT (assuming lamp wiring is passed through digital switch 0)
                MacroInterface.SetOutput(eOutputTarget.DUT_MAINS_SUPPLY, true);  //switch on DUT


                //wait for 2000 milliseconds after switching to be sure to have a stable DALI bus
                Thread.Sleep(2000);


                //set arc power level to 254 and query current lamp failure state
                macro.Add(new cCommandElement2(eDALICommand2.DAPC, 254)); //uses the global selected address given by the ProbitBench
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_ACTUAL_LEVEL, null));
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_LAMP_FAILURE, null));
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_STATUS, null));
                result = MacroInterface.SendMacroAndWaitForAnswer(macro); //blocking call
                //DO SOMETHING WITH LATEST ANSWER OF MACRO (HERE QUERY STATUS ANSWER)


                //disconnect lamp from DUT
                MacroInterface.SetOutput(eOutputTarget.DIGITAL_SWITCH_0, false);


                //wait for 5000 milliseconds
                Thread.Sleep(5000);


                //query for lamp failure
                macro.Clear();
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_ACTUAL_LEVEL, null));
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_LAMP_FAILURE, null));
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_STATUS, null));
                result = MacroInterface.SendMacroAndWaitForAnswer(macro); //blocking call
                //DO SOMETHING WITH LATEST ANSWER OF MACRO (HERE QUERY STATUS ANSWER)


                //connect lamp to DUT
                MacroInterface.SetOutput(eOutputTarget.DIGITAL_SWITCH_0, true);


                //wait for 5000 milliseconds
                Thread.Sleep(5000);


                //query for lamp failure
                macro.Clear();
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_ACTUAL_LEVEL, null));
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_LAMP_FAILURE, null));
                macro.Add(new cCommandElement2(eDALICommand2.QUERY_STATUS, null));
                result = MacroInterface.SendMacroAndWaitForAnswer(macro); //blocking call
                //DO SOMETHING WITH LATEST ANSWER OF MACRO (HERE QUERY STATUS ANSWER)


                //wait for 2000 milliseconds before switching off to be sure last macro has been executed on the DALI bus
                Thread.Sleep(2000);


                //Set all switches to default operation after procudure
                MacroInterface.SetOutput(eOutputTarget.DUT_MAINS_SUPPLY, false);  //switch off DUT
                MacroInterface.SetOutput(eOutputTarget.DIGITAL_SWITCH_0, false); //disconnect lamp from DUT (assuming lamp wiring is passed through digital switch 0)
                MacroInterface.SetOutput(eOutputTarget.DALI_TERMINAL_POLARITY_INVERTED, false); //disable DALI bus terminal invertion
                MacroInterface.SetOutput(eOutputTarget.DALI_TERMINAL_DUT_CONNECTED, false); //disconnect DALI bus from DUT
                MacroInterface.SetOutput(eOutputTarget.DALI_BUS_POWER, false); //disable DALI BPS
            }
        }
        
        #endregion
        
        #region example: setting outputs

        bool m_bps_16V_250mA = true; //variable to remember which BPS setting is currently used
        private void SetBPSButton_Click(object sender, System.EventArgs e)
        {
            bool set_ok = false;

            if (m_bps_16V_250mA)
            {
                set_ok = MacroInterface.SetOutput(eOutputTarget.DALI_BPS_VOLTAGE, (double)12.0); // set bps voltage of test system to 12V
                set_ok &= MacroInterface.SetOutput(eOutputTarget.DALI_BPS_CURRENT, (double)8.0); // set bps current of test system to 8mA

                m_bps_16V_250mA = false;
                SetBPSButton.Text = "Set BPS to 16 V / 250 mA"; //change button text accordingly
            }
            else
            {
                set_ok = MacroInterface.SetOutput(eOutputTarget.DALI_BPS_VOLTAGE, (double)16.0); // set bps voltage of test system to 16V
                set_ok &= MacroInterface.SetOutput(eOutputTarget.DALI_BPS_CURRENT, (double)250.0); // set bps current of test system to 250mA

                m_bps_16V_250mA = true;
                SetBPSButton.Text = "Set BPS to 12 V / 8 mA"; //change button text accordingly
            }

            if (!set_ok)
            {
                MessageBox.Show("Setup of BPS failed!"); //in case of setup has failed
            }
        }

        #endregion

        #region example: reading inputs

        private void ReadInputsButton_Click(object sender, System.EventArgs e)
        {
            object read_value = null; // return object needed for the read operations, casting has to be done after read

            //target variables used later for storing the read value
            bool? digital_in_0 = null;
            bool? digital_in_1 = null;
            double? analog_in_0 = null;
            double? analog_in_1 = null;
            double? dali_bus_voltage = null;
            double? light_meter = null;

            //read digital 0 (be aware that code is using zero-based index and refers to Digital In 1 of the test system)
            if (MacroInterface.GetInput(eInputTarget.DIGITAL_IN_0, ref read_value))
            {
                if (read_value != null)
                {
                    digital_in_0 = (bool)read_value;
                }
            }

            //read digital 1 (be aware that code is using zero-based index and refers to Digital In 2 of the test system)
            if (MacroInterface.GetInput(eInputTarget.DIGITAL_IN_1, ref read_value))
            {
                if (read_value != null)
                {
                    digital_in_1 = (bool)read_value;
                }
            }

            //read analog 0 (be aware that code is using zero-based index and refers to Analog In 1 of the test system)
            if (MacroInterface.GetInput(eInputTarget.ANALOG_IN_0, ref read_value))
            {
                if (read_value != null)
                {
                    analog_in_0 = (double)read_value;
                }
            }

            //read analog 1 (be aware that code is using zero-based index and refers to Analog In 2 of the test system)
            if (MacroInterface.GetInput(eInputTarget.ANALOG_IN_1, ref read_value))
            {
                if (read_value != null)
                {
                    analog_in_1 = (double)read_value;
                }
            }

            //read bus voltage measured at the terminals of the test system
            if (MacroInterface.GetInput(eInputTarget.DALI_BUS_VOLTAGE, ref read_value))
            {
                if (read_value != null)
                {
                    dali_bus_voltage = (double)read_value;
                }
            }

            //read light value measured with external light meter connected to the ProbitBench
            if (MacroInterface.GetInput(eInputTarget.LIGHT_METER, ref read_value))
            {
                if (read_value != null)
                {
                    light_meter = (double)read_value;
                }
            }

            #region report read values
            string output_text = "Read inputs:\r\n";
            if (digital_in_0 != null) output_text += "\r\nDigital In 0: " + digital_in_0;
            else output_text += "\r\nDigital In 0: n.a.";
            if (digital_in_1 != null) output_text += "\r\nDigital In 1: " + digital_in_1;
            else output_text += "\r\nDigital In 1: n.a.";
            if (analog_in_0 != null) output_text += "\r\nAnalog In 0: " + analog_in_0 + " V";
            else output_text += "\r\nAnalog In 0: n.a.";
            if (analog_in_1 != null) output_text += "\r\nAnalog In 1: " + analog_in_1 + " V";
            else output_text += "\r\nAnalog In 1: n.a.";
            if (dali_bus_voltage != null) output_text += "\r\nDALI bus voltage: " + dali_bus_voltage + " V";
            else output_text += "\r\nDALI bus voltage: n.a.";
            if (light_meter != null) output_text += "\r\nLight meter: " + light_meter + " lx";
            else output_text += "\r\nLight meter: n.a.";
            MessageBox.Show(output_text);
            #endregion
        }

        #endregion

        #region example: reading test parameters

        private void ReadTestParametersButton_Click(object sender, System.EventArgs e)
        {
            // return object needed for the read operations, casting has to be done after read
            object read_parameter = null;

            //target variables used later for storing the read value
            bool? is_bus_powered_device = null;
            bool? is_bps_unit_integrated = null;
            double? bps_voltage = null;
            double? bps_max_current = null;

            //read test parameter if device is a bus powered device
            if (MacroInterface.GetTestParameters(eTestParameterNames.IS_BUS_POWERED_DEVICE, ref read_parameter))
            {
                if (read_parameter != null)
                {
                    try
                    {
                        is_bus_powered_device = (bool)read_parameter;
                    }
                    catch
                    {
                        is_bus_powered_device = null;
                    }
                }
            }

            //read test parameter if device has a bus power supply integrated
            if (MacroInterface.GetTestParameters(eTestParameterNames.IS_BPS_UNIT_INTEGRATED, ref read_parameter))
            {
                if (read_parameter != null)
                {
                    try
                    {
                        is_bps_unit_integrated = (bool)read_parameter;
                    }
                    catch
                    {
                        is_bps_unit_integrated = null;
                    }
                }
            }

            //read test parameter of DUT bus power supply voltage (in case of there is one)
            if (MacroInterface.GetTestParameters(eTestParameterNames.BPS_VOLTAGE, ref read_parameter))
            {
                if (read_parameter != null)
                {
                    try
                    {
                        bps_voltage = (double)read_parameter;
                    }
                    catch
                    {
                        bps_voltage = null;
                    }
                }
            }

            //read test parameter of DUT bus power supply maximum current (in case of there is one)
            if (MacroInterface.GetTestParameters(eTestParameterNames.BPS_MAXIMUM_CURRENT, ref read_parameter))
            {
                if (read_parameter != null)
                {
                    try
                    {
                        bps_max_current = (double)read_parameter;
                    }
                    catch
                    {
                        bps_max_current = null;
                    }
                }
            }

            #region report read parameters
            string output_text = "Read test parameters:\r\n";
            if (is_bus_powered_device != null) output_text += "\r\nBus powered device: " + is_bus_powered_device;
            else output_text += "\r\nBus powered device: n.a.";
            if (is_bps_unit_integrated != null) output_text += "\r\nBPS unit integrated: " + is_bps_unit_integrated;
            else output_text += "\r\nBPS unit integrated: n.a.";
            if (bps_voltage != null) output_text += "\r\nBPS voltage: " + bps_voltage + " V";
            else output_text += "\r\nBPS voltage: n.a.";
            if (bps_max_current != null) output_text += "\r\nBPS maximum current: " + bps_max_current + " mA";
            else output_text += "\r\nBPS maxium current: n.a.";
            MessageBox.Show(output_text);
            #endregion
        }

        #endregion

        #region example: sniffer related functions

        private System.Windows.Forms.Timer m_sniffer_result_update_timer = null;
        private int number_of_sniffer_entries_processed = 0;

        private void Sniffer1Button_Click(object sender, System.EventArgs e)
        {
            if (m_sniffer_result_update_timer == null)
            {
                //sniffer not started => start sniffer

                //stop previous running sniffer (just in case)
                SnifferInterface.StopSniffer();

                //clear sniffer output listbox
                List<string> columns = new List<string>();
                columns.Add("Time");
                columns.Add("Frame");
                SnifferResultBox.InitGrid(10000, columns, null);
                //or when grid already initialized: SnifferResultBox.ClearRows();
                number_of_sniffer_entries_processed = 0;

                //start the sniffer
                SnifferInterface.StartSniffer();

                //setup and start sniffer result update
                m_sniffer_result_update_timer = new System.Windows.Forms.Timer();
                m_sniffer_result_update_timer.Tick += new System.EventHandler(SnifferResultUpdateTick);
                m_sniffer_result_update_timer.Interval = 500;
                m_sniffer_result_update_timer.Start();

                //change button text
                Sniffer1Button.Text = "Stop sniffer";
            }
            else
            {
                //sniffer already started => stop sniffer

                //stop running sniffer update timer
                m_sniffer_result_update_timer.Stop();
                m_sniffer_result_update_timer = null;

                //stop running sniffer
                SnifferInterface.StopSniffer();

                //change button text
                Sniffer1Button.Text = "Start sniffer";
            }
        }

        private bool row_toggle = false;
        private void SnifferResultUpdateTick(object sender, System.EventArgs e)
        {
            //stop the update timer for retrieving the sniffer results
            m_sniffer_result_update_timer.Stop();
                        
            //check if any new sniffer results are available
            List<cSnifferResult> current_sniffer_protocol = SnifferInterface.GetSnifferProtocol();
            
            for (int i = number_of_sniffer_entries_processed; i < current_sniffer_protocol.Count; i++)
            {
                //try to retrieve the next sniffer results 
                cSnifferResult next_sniffer_result = current_sniffer_protocol[i];
                number_of_sniffer_entries_processed++;

                if (next_sniffer_result != null)
                {
                    bool add_line = true;
                    string[] new_line = new string[2] { "-", "-" };

                    //evaluate sniffer result according its type and generate a proper string for the sniffer output
                    switch (next_sniffer_result.m_result)
                    {
                        case eSnifferResult.FRAME:
                            {
                                new_line[0] = PrintTime(next_sniffer_result.m_timestamp_1);

                                string frame_str = "";
                                if (next_sniffer_result.m_frame_size <= 8) frame_str = next_sniffer_result.m_frame.ToString("X2");
                                else if (next_sniffer_result.m_frame_size <= 16) frame_str = next_sniffer_result.m_frame.ToString("X4");
                                else if (next_sniffer_result.m_frame_size <= 24) frame_str = next_sniffer_result.m_frame.ToString("X6");
                                else frame_str = next_sniffer_result.m_frame.ToString("X8");

                                new_line[1] = "0x" + frame_str + " (" + next_sniffer_result.m_frame_size + " bits)";
                            }
                            break;

                        case eSnifferResult.BIT_TIME_VIOLATION:
                            {
                                new_line[0] = PrintTime(next_sniffer_result.m_timestamp_1);
                                new_line[1] = "Bit time violation";
                            }
                            break;

                        case eSnifferResult.SYSTEM_FAILURE:
                            {
                                new_line[0] = PrintTime(next_sniffer_result.m_timestamp_1);
                                new_line[1] = "System failure";
                            }
                            break;

                        case eSnifferResult.IDLE:
                            {
                                new_line[0] = PrintTime(next_sniffer_result.m_timestamp_1);
                                new_line[1] = "Idle";
                            }
                            break;

                        default:
                            add_line = false;
                            break;
                    }

                    //add new line to the sniffer output (if not empty)
                    if (add_line)
                    {
                        if (row_toggle)
                        {
                            SnifferResultBox.AddRow(null, new_line);
                        }
                        else
                        {
                            SnifferResultBox.AddRow(null, Color.AliceBlue, Color.Black, null, null, new_line);
                        }

                        row_toggle = !row_toggle;
                    }
                }
            }

            //re-enable the update timer
            m_sniffer_result_update_timer.Start();
        }

        private string PrintTime(ulong t)
        {
            //create output string for timestamp used by a sniffer result

            ulong h = t / 3600000000;
            t = t % 3600000000;
            ulong m = t / 60000000;
            t = t % 60000000;
            ulong s = t / 1000000;
            t = t % 1000000;
            ulong ms = t / 1000;
            t = t % 1000;
            ulong us = t;


            string hh;
            if (h < 10) hh = "0" + h.ToString() + ":";
            else hh = h.ToString() + ":";

            string mm;
            if (m < 10) mm = "0" + m.ToString() + ":";
            else mm = m.ToString() + ":";

            string ss;
            if (s < 10) ss = "0" + s.ToString() + ".";
            else ss = s.ToString() + ".";

            string msmsms;
            if (ms < 10) msmsms = "00" + ms.ToString() + "::";
            else if (ms < 100) msmsms = "0" + ms.ToString() + "::";
            else msmsms = ms.ToString() + "::";

            string ususus;
            if (us < 10) ususus = "00" + us.ToString();
            else if (us < 100) ususus = "0" + us.ToString();
            else ususus = us.ToString();

            return hh + mm + ss + msmsms + ususus;
        }

        #endregion        
    }
}
